/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation.investigations;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.ListSelectionDialog;

import com.ericsson.otp.erlang.OtpErlangObject;
import com.refactorerl.ui.communication.exceptions.CommunicationException;
import com.refactorerl.ui.communication.exceptions.ConnectionException;
import com.refactorerl.ui.communication.exceptions.RequestException;
import com.refactorerl.ui.logic.investigations.Investigation;
import com.refactorerl.ui.logic.investigations.InvestigationNode;
import com.refactorerl.ui.logic.investigations.InvestigationsFactory;
import com.refactorerl.ui.logic.investigations.NodeQueryRequest;
import com.refactorerl.ui.logic.investigations.SaveInvestigationRequest;
import com.refactorerl.ui.presentation.Activator;

public class InvestigationHelper {

	public static List<InvestigationNode> showSelectResultNodeDialog(String query, InvestigationNode parentNode,
			int caretPosition, IWorkbenchPage activePage) {
		try {
			String filePath = parentNode.filePath;
			int position = parentNode.offset + caretPosition;

			List<OtpErlangObject> graphNodes = Activator.executeRequest(new NodeQueryRequest(filePath, position, query,
					Activator.getProxy(), null), activePage.getWorkbenchWindow().getShell());

			List<InvestigationNode> invNodes = new ArrayList<>();
			for (OtpErlangObject gn : graphNodes) {
			
				invNodes.add(InvestigationsFactory.createNewInvestigationNode(gn, parentNode.pid, query,
						Activator.getProxy(), null));
			}

			IStructuredContentProvider contentProvider = new IStructuredContentProvider() {

				private List<InvestigationNode> input;

				@Override
				public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
					this.input = (List<InvestigationNode>) newInput;
				}

				@Override
				public void dispose() {
			
				}

				@Override
				public Object[] getElements(Object inputElement) {
					return input.toArray();

				}
			};

			ILabelProvider labelProvider = new LabelProvider() {
				@Override
				public String getText(Object element) {
					InvestigationNode node = (InvestigationNode) element;
					return node.fileItemLabel + " (" + node.filePath + ":" + node.offset + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				}
			};

			ListSelectionDialog dialog = new ListSelectionDialog(activePage.getWorkbenchWindow().getShell(), invNodes,
					contentProvider, labelProvider, Messages.InvestigationHelper_3);
			if (dialog.open() == dialog.OK) {
				List<InvestigationNode> results = new ArrayList<>();
				for (Object o : dialog.getResult()) {
					results.add((InvestigationNode) o);
				}
				return results;
			}

		} catch (RequestException | ConnectionException | CommunicationException e) {
			Activator.showErrorDialog(e, activePage.getWorkbenchWindow().getShell());
		}
		return null;
	}

	public static Investigation showNewInvestigation(Investigation investigation, IWorkbenchPage activePage)
			throws PartInitException, RequestException, ConnectionException, CommunicationException {

		activePage.openEditor(new InvestigationEditorInput(investigation), "com.refactorerl.ui.investigationEditor"); //$NON-NLS-1$

		Activator.executeRequest(
				SaveInvestigationRequest.createSaveInvestigationRequest(investigation, Activator.getProxy(), null),
				activePage.getWorkbenchWindow().getShell());

		InvestigationsView view = (InvestigationsView) activePage.showView("com.refactorerl.ui.investigationsView"); //$NON-NLS-1$	
		view.refreshList(); 
		
		return investigation;

	}

}
