/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation.dupcode;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModel;
import org.eclipse.jface.text.source.AnnotationPainter;
import org.eclipse.jface.text.source.CompositeRuler;
import org.eclipse.jface.text.source.IAnnotationAccess;
import org.eclipse.jface.text.source.IOverviewRuler;
import org.eclipse.jface.text.source.ISharedTextColors;
import org.eclipse.jface.text.source.LineNumberRulerColumn;
import org.eclipse.jface.text.source.OverviewRuler;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.editors.text.EditorsUI;
import org.eclipse.ui.internal.editors.text.EditorsPlugin;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.texteditor.AnnotationPreference;
import org.eclipse.ui.texteditor.SourceViewerDecorationSupport;

import com.refactorerl.ui.common.FilePosition;
import com.refactorerl.ui.presentation.codebrowser.StringInput;
import com.refactorerl.ui.presentation.codebrowser.StringStorage;

/*
 * Note: annotations can also be create on an IResource (such as an IFile), the
 * responsible editor will show it
 */

/**
 * Creates a naive two pane editor for showing duplications on its two sides.
 * 
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 */
public class TwinPaneEditor extends EditorPart {

	private TwinPaneEditorInput input;

	private List<EditorModel> editorModels = new ArrayList<>();

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		setSite(site);
		setInput(input);
	}

	@Override
	protected void setInput(IEditorInput input) {
		super.setInput(input);
		this.input = (TwinPaneEditorInput) input;
	}
	
	@Override
	public void createPartControl(Composite parent) {
		Collection<StringInput> strInputs = input.getStringInputs();

		SashForm form = new SashForm(parent,SWT.HORIZONTAL);
		form.setLayout(new FillLayout());
		
		List<Composite> cmps = new ArrayList<>();
		for (StringInput in : strInputs) {
			Composite cmp = new Composite(form, SWT.NONE);
			GridLayout layout = new GridLayout(1, true);			
			cmp.setLayout(layout);
			cmps.add(cmp);

			Label lab = new Label(cmp, SWT.BORDER);
			lab.setText(in.getName());	
			lab.setData(new GridData(SWT.FILL, SWT.BEGINNING, true, true));
			
			EditorModel em = new EditorModel();
			em.init(cmp, in);
			
			FilePosition fp = input.getFilePositions().get(in);
			em.highlight(fp.startPosition -1, fp.endPosition);
			
			editorModels.add(em);		
			em.sourceViewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));			
		}
		
		StringBuilder sb = new StringBuilder();  
		String delim = ""; //$NON-NLS-1$
		for (StringInput in : strInputs) {
			String orig = input.getFilePositions().get(in).toString();
			String shortened = orig.substring(orig.lastIndexOf('/') +1); 
		    sb.append(delim).append(shortened);
		    delim = " | "; //$NON-NLS-1$
		}
								
		setPartName(sb.toString()); 

	}

	private class EditorModel {
		public SourceViewer sourceViewer;
		public SourceViewerDecorationSupport sds;
		public IDocument document;
		public AnnotationModel annotationModel;

		public ISharedTextColors sharedColors;
		public IOverviewRuler overviewRuler;
		public CompositeRuler ruler;
		public IAnnotationAccess ac;
		public AnnotationPainter ap;
		public Position position;
		public Annotation annotation;

		public void dispose() {
			sds.dispose();
		}

		public void highlight(int start, int end) {
			sourceViewer.getAnnotationModel().removeAnnotation(annotation);

			annotation = new Annotation(false);
			annotation.setType("com.refactorerl.ui.dupCodeMarker"); //$NON-NLS-1$
			position = new Position(start, end - start);
			sourceViewer.getAnnotationModel().addAnnotation(annotation, position);

			sourceViewer.revealRange(start, end - start);
		}

		public void init(Composite parent, StringInput input) {
			int VERTICAL_RULER_WIDTH = 12;

			int styles = SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI | SWT.BORDER | SWT.FULL_SELECTION;
			sharedColors = EditorsUI.getSharedTextColors();
			overviewRuler = new OverviewRuler(null, VERTICAL_RULER_WIDTH, sharedColors);
			ruler = new CompositeRuler(VERTICAL_RULER_WIDTH);
			ruler.addDecorator(0, new LineNumberRulerColumn());
			document = new Document();
			document.set(((StringStorage) input.getStorage()).getString());
			annotationModel = new AnnotationModel();
			annotationModel.connect(document);
			sourceViewer = new SourceViewer(parent, ruler, overviewRuler, true, styles);
			sourceViewer.configure(new SourceViewerConfiguration());
			sds = new SourceViewerDecorationSupport(sourceViewer, overviewRuler, null, sharedColors);

			sds.setAnnotationPainterPreferenceKeys("com.refactorerl.ui.dupCodeMarker", //$NON-NLS-1$
					"com.refactorerl.ui.dupCodeMarker.color", //$NON-NLS-1$
					"com.refactorerl.ui.dupCodeMarker.text", //$NON-NLS-1$
					"com.refactorerl.ui.dupCodeMarker.overview", 0); //$NON-NLS-1$
			sds.setAnnotationPreference(new AnnotationPreference("com.refactorerl.ui.dupCodeMarker", //$NON-NLS-1$
					"com.refactorerl.ui.dupCodeMarker.color", //$NON-NLS-1$
					"com.refactorerl.ui.dupCodeMarker.text", //$NON-NLS-1$
					"com.refactorerl.ui.dupCodeMarker.overview", 0)); //$NON-NLS-1$
			sds.install(EditorsPlugin.getDefault().getPreferenceStore());

			sourceViewer.setDocument(document, annotationModel);

		

			ac = new IAnnotationAccess() {

				@Override
				public boolean isTemporary(Annotation annotation) {

					return true;
				}

				@Override
				public boolean isMultiLine(Annotation annotation) {
					return true;
				}

				@Override
				public Object getType(Annotation annotation) {

					return annotation.getType();
				}
			};

			ap = new AnnotationPainter(sourceViewer, ac);
			ap.addHighlightAnnotationType("com.refactorerl.ui.dupCodeMarker"); //$NON-NLS-1$
			ap.setAnnotationTypeColor("com.refactorerl.ui.dupCodeMarker", //$NON-NLS-1$
					new Color(PlatformUI.getWorkbench().getDisplay(), new RGB(250, 228, 165)));
			sourceViewer.addPainter(ap);
			sourceViewer.addTextPresentationListener(ap);

		}

	}

	public void dispose() {
		for (EditorModel em : editorModels) {
			em.dispose();
		}	
	}

	public void doSave(IProgressMonitor monitor) {
	}

	public void doSaveAs() {
	}

	public boolean isDirty() {
		return false;
	}

	public boolean isSaveAsAllowed() {
		return false;
	}

	public void setFocus() {

	}

}