/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation.dependency;

import java.util.Arrays;
import java.util.List;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.refactorerl.ui.common.Pair;
import com.refactorerl.ui.communication.exceptions.ReferlException;
import com.refactorerl.ui.logic.GetFileFromServerRequest;
import com.refactorerl.ui.logic.dependency.DependencyRequest;
import com.refactorerl.ui.logic.dependency.DependencyRequestFactory;
import com.refactorerl.ui.presentation.Activator;

/**
 * Defines the widgets present in the module dependency panels.
 * 
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 */
class ModuleControlPanel extends ModuleFunctionControlPanel {
	private Shell shell;

	public ModuleControlPanel(Composite parent, FormToolkit toolkit,
			List<Control> hideWhenErlSelected,
			List<Control> hideWhenHTMLSelected,
			List<Control> hideWhenModuleBlockSelected,
			List<String> moduleProposals) {
		super(parent, toolkit, hideWhenErlSelected, hideWhenHTMLSelected,
				hideWhenModuleBlockSelected, moduleProposals,
				Messages.ModuleControlPanel_0, Messages.ModuleControlPanel_1,
				Messages.ModuleControlPanel_2);
		this.shell = parent.getShell();
	}

	@Override
	protected String makeDependencyRequest(
			DependencyRequestFactory.DepOutputFormat outputFormat,
			DependencyRequestFactory.DepType type, String fileName) {

		class Input {
			public void init() {
				startsStr = Arrays.asList(starts.getItems());
				excludedsStr = Arrays.asList(excludeds.getItems());
				excludedLeavesStr = Arrays.asList(excludedLeaves.getItems());
				excludeOtp = excludeOtpButton.getSelection();
			}

			List<String> startsStr;
			List<String> excludedsStr;
			List<String> excludedLeavesStr;
			boolean excludeOtp;
		}

		final Input input = new Input();
		parent.getDisplay().syncExec(new Runnable() {

			@Override
			public void run() {
				input.init();
			}
		});

		List<String> startsStr = input.startsStr;
		List<String> excludedsStr = input.excludedsStr;
		List<String> excludedLeavesStr = input.excludedLeavesStr;
		boolean excludeOtp = input.excludeOtp;

		Pair<DependencyRequest, String> reqPathPair = null;
		try {
			switch (outputFormat) {
			case DOT:
				reqPathPair = DependencyRequestFactory
						.createSVGDrawModuleRequest(type, fileName, excludeOtp,
								startsStr, excludedsStr, excludedLeavesStr,
								Activator.getProxy(), null);
				break;

			case HTML:
				reqPathPair = DependencyRequestFactory
						.createHTMLDrawModuleRequest(type, fileName,
								excludeOtp, startsStr, excludedsStr,
								excludedLeavesStr, Activator.getProxy(), null);
				break;
			case ERL:
				return (String) Activator.executeRequest(DependencyRequestFactory
						.createPrintModuleRequest(startsStr,
								Activator.getProxy(), null), shell);
			}
		} catch (ReferlException e) {
			Activator.showErrorDialog(e, shell);
		}
		DependencyRequest request = reqPathPair.key;
		Activator.executeRequest(request, shell);

		String filePathWithExtension = reqPathPair.value;
		// FIXME encoding issues are possible
		String fileContent = new String(Activator.executeRequest(
				new GetFileFromServerRequest(filePathWithExtension, Activator
						.getProxy(), null), shell));

		return fileContent;

	}

}