/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation.dependency;

import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.refactorerl.ui.common.Pair;
import com.refactorerl.ui.communication.exceptions.CommunicationException;
import com.refactorerl.ui.communication.exceptions.ConnectionException;
import com.refactorerl.ui.communication.exceptions.RequestException;
import com.refactorerl.ui.logic.GetFileFromServerRequest;
import com.refactorerl.ui.logic.dependency.DependencyRequest;
import com.refactorerl.ui.logic.dependency.DependencyRequestFactory;
import com.refactorerl.ui.presentation.Activator;

/**
 * Defines the widgets present in the module block dependency panels.
 * 
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 */
class ModuleBlockControlPanel extends ControlPanel {
	private Shell shell;
	private Composite cmp;
	private org.eclipse.swt.widgets.List dirs;
	private Text regexText;
	private Button regexButton;

	public ModuleBlockControlPanel(Composite parent, FormToolkit toolkit,
			List<Control> hideWhenErlSelected,
			List<Control> hideWhenHTMLSelected,
			List<Control> hideWhenModuleBlockSelected,
			List<String> directoryProposals) {
		super(parent, toolkit, hideWhenErlSelected, hideWhenHTMLSelected,
				hideWhenModuleBlockSelected);
		this.shell = parent.getShell();

		cmp = toolkit.createComposite(parent);
		cmp.setLayout(new GridLayout(2, false));

		Button dirButton = toolkit.createButton(cmp,
				Messages.ModuleBlockControlPanel_0, SWT.RADIO);

		dirs = createListOnGridLayout(cmp, directoryProposals, dirButton);

		regexButton = toolkit.createButton(cmp,
				Messages.ModuleBlockControlPanel_1, SWT.RADIO);

		configureGridData(regexButton, false);

		regexText = toolkit.createText(cmp, "", SWT.SINGLE | SWT.BORDER); //$NON-NLS-1$

		configureGridData(regexText, true);
	}

	@Override
	public Composite getComposite() {
		return cmp;

	}

	@Override
	protected String makeDependencyRequest(
			DependencyRequestFactory.DepOutputFormat outputFormat,
			DependencyRequestFactory.DepType type, String fileName) {

		class Input {
			// String filePath;
			boolean regexSelected;
			String regexTextStr;
			List<String> dirsStr;

			public void init() {
				regexSelected = regexButton.getSelection();
				dirsStr = Arrays.asList(dirs.getItems());
				regexTextStr = regexText.getText();
			}
		}

		final Input input = new Input();
		parent.getDisplay().syncExec(new Runnable() {

			@Override
			public void run() {
				input.init();
			}
		});

		boolean regexSelected = input.regexSelected;
		String regexTextStr = input.regexTextStr;
		List<String> dirsStr = input.dirsStr;

		Pair<DependencyRequest, String> reqPathPair = null;
		try {
			switch (outputFormat) {
			case DOT:
				if (regexSelected) {

					reqPathPair = DependencyRequestFactory
							.createSVGDrawModuleBlockRequest(type, fileName,
									regexTextStr, Activator.getProxy(), null);

				} else {
					reqPathPair = DependencyRequestFactory
							.createSVGDrawModuleBlockRequest(type, fileName,
									dirsStr, Activator.getProxy(), null);
				}
				break;
			case ERL:
				parent.getDisplay().syncExec(new Runnable() {

					@Override
					public void run() {
						MessageDialog.openInformation(parent.getShell(),
								Messages.ModuleBlockControlPanel_4,
								Messages.ModuleBlockControlPanel_5
										+ Messages.ModuleBlockControlPanel_6);
					}
				});
				return null;
			case HTML:
				parent.getDisplay().syncExec(new Runnable() {

					@Override
					public void run() {
						MessageDialog.openInformation(parent.getShell(),
								Messages.ModuleBlockControlPanel_7,
								Messages.ModuleBlockControlPanel_8
										+ Messages.ModuleBlockControlPanel_9);
					}
				});
				return null;
			}
		} catch (RequestException | ConnectionException
				| CommunicationException e) {
			Activator.showErrorDialog(e, shell);
		}

		DependencyRequest request = reqPathPair.key;
		Activator.executeRequest(request, shell);

		String filePathWithExtension = reqPathPair.value;

		// FIXME encoding issues are possible
		String fileContent = new String(Activator.executeRequest(
				new GetFileFromServerRequest(filePathWithExtension, Activator
						.getProxy(), null), shell));

		return fileContent;

	}

}