/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.NotEnabledException;
import org.eclipse.core.commands.NotHandledException;
import org.eclipse.core.commands.Parameterization;
import org.eclipse.core.commands.ParameterizedCommand;
import org.eclipse.core.commands.common.NotDefinedException;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.progress.UIJob;
import org.eclipse.ui.services.IServiceLocator;

import com.refactorerl.ui.logic.IsFilePresentRequest;
import com.refactorerl.ui.presentation.filelist.FileListView;

/**
 * Will handle the changes of those local resources, which were added to the
 * RefactorErl database by calling a database synchronization request.
 * 
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 */
public class ReferlResourceChangeListener implements IResourceChangeListener {

	IWorkbenchWindow wb;

	public ReferlResourceChangeListener(IWorkbenchWindow wb) {
		this.wb = wb;
	}

	@Override
	public void resourceChanged(IResourceChangeEvent event) {
		boolean databaseSyncNeeded = false;
		List<String> dropablePaths = new ArrayList<>();

		IResourceDelta[] resources = computeResourceLeaves(event);
		for (IResourceDelta rd : resources) {

			if (!(rd.getKind() == IResourceDelta.CHANGED || rd.getKind() == IResourceDelta.REMOVED))
				continue;

			String loc = rd.getResource().getLocation().toPortableString();

			boolean fileInDb = Activator.executeRequest(
					new IsFilePresentRequest(loc, Activator.getProxy(), null),
					wb.getShell());

			if (!fileInDb)
				continue;

			databaseSyncNeeded = true; // once set, stays true

			if (rd.getKind() == IResourceDelta.REMOVED)
				dropablePaths.add(loc);

		}

		if (databaseSyncNeeded) {
			IServiceLocator serviceLocator = PlatformUI.getWorkbench();
			final ICommandService commandService = (ICommandService) serviceLocator
					.getService(ICommandService.class);
			final IHandlerService handlerService = (IHandlerService) serviceLocator
					.getService(IHandlerService.class);
			try {
				// NOTE same code is in AddDropResourceHandler
				for (String path : dropablePaths) {
					Command command = commandService
							.getCommand("com.refactorerl.ui.dropFile"); //$NON-NLS-1$
					Parameterization p = new Parameterization(
							command.getParameter("com.refactorerl.ui.dropFile.path"), path); //$NON-NLS-1$
					final ParameterizedCommand pc = new ParameterizedCommand(command,
							new Parameterization[] { p });
					new UIJob("execute") { //$NON-NLS-1$
						@Override
						public IStatus runInUIThread(IProgressMonitor monitor) {

							try {
								handlerService.executeCommand(pc, null);
							} catch (ExecutionException | NotDefinedException
									| NotEnabledException | NotHandledException e) {
								Activator.showErrorDialog(e, wb.getShell());
							}
							return Status.OK_STATUS;
						}

					}.schedule();
				}
				new UIJob("execute") { //$NON-NLS-1$

					@Override
					public IStatus runInUIThread(IProgressMonitor monitor) {
						try {
							commandService.getCommand(
									"com.refactorerl.ui.dbSync") //$NON-NLS-1$
									.executeWithChecks(new ExecutionEvent());
						} catch (ExecutionException | NotDefinedException
								| NotEnabledException | NotHandledException e) {
							Activator.showErrorDialog(e, wb.getShell());
						} 
						
						// FIXME won't handle views in other perspective
						// activeWB.getPages() won't work for this either
						final FileListView fileListView = (FileListView) wb.getActivePage().findView(
								"com.refactorerl.ui.fileListView"); //$NON-NLS-1$

						if (fileListView != null)
							fileListView.reload();
						
						return Status.OK_STATUS;
					}

				}.schedule();
				//	commandService.getCommand("com.refactorerl.ui.dbSync").executeWithChecks(new ExecutionEvent()); //$NON-NLS-1$
			} catch (/*ExecutionException | */NotDefinedException
					/*| NotEnabledException | NotHandledException */ e) {
				Activator.showErrorDialog(e, wb.getShell());
			}
		}

	}

	/**
	 * Traverses the tree formed by IResourceDelta nodes.
	 * 
	 * @param event
	 * @return
	 */
	public static IResourceDelta[] computeResourceLeaves(
			IResourceChangeEvent event) {
		List<IResourceDelta> leaves = new ArrayList<>();

		IResourceDelta root = event.getDelta();
		Stack<IResourceDelta> stack = new Stack<>();
		stack.push(root);
		while (!stack.isEmpty()) {
			IResourceDelta current = stack.pop();
			IResourceDelta[] children = current.getAffectedChildren();
			if (children.length == 0)
				leaves.add(current);
			for (IResourceDelta rd : children) {
				stack.add(rd);
			}
		}

		IResourceDelta[] resources = new IResourceDelta[leaves.size()];
		int i = 0;
		for (IResourceDelta rd : leaves) {

			resources[i] = rd;
			++i;
		}

		return resources;
	}

}
