/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.logic.dupcode;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AlgorithmData {
	public enum DataType {
		ENUM, ATOM, ATOMS, INT, FLOAT, BOOL
	};

	public enum EnumType {
		STRING, ATOM
	};

	final static private Map<DataType, Class<?>> typeTranslations = new HashMap<>();
	{
		typeTranslations.put(DataType.ENUM, String.class);
		typeTranslations.put(DataType.ATOM, String.class);
		typeTranslations.put(DataType.ATOMS, List.class);
		typeTranslations.put(DataType.INT, Integer.class);
		typeTranslations.put(DataType.FLOAT, Double.class);
		typeTranslations.put(DataType.BOOL, Boolean.class);
	}
	// static public Map<DataType, Class<?>> getTypeTranslations(){
	// return Collections.unmodifiableMap(typeTranslations);
	// }

	final public String key;
	final public String label;
	final public DataType type;
	final public String defaultValue;

	final public String[] enumOptions;
	final public EnumType enumType;

	private Object value = null;

	public AlgorithmData(String key, String label, DataType type,
			String defaultValue) {
		this(key, label, type, defaultValue, null, null);

	}

	public AlgorithmData(String key, String label, DataType type,
			String defaultValue, String[] enumOptions, EnumType enumType) {
		super();
		this.key = key;
		this.label = label;
		this.type = type;
		this.defaultValue = defaultValue;
		this.enumOptions = enumOptions;
		this.enumType = enumType;

		setDefaultValue();
	}

	/**
	 * Sets the default value as the value of this AlgorithmData.
	 * 
	 */
	public void setDefaultValue() {
		switch (type) {
		case ATOM:
		case ENUM:
			setValue((String) defaultValue);
			break;
		case ATOMS:
			// assumes only empty list will arrive as default
			setValue(new ArrayList<>()); 
			break;
		case BOOL:
			setValue(getBooleanDefault());
			break;
		case FLOAT:
			setValue(Double.parseDouble(defaultValue));
			break;
		case INT:
			setValue(Integer.parseInt(defaultValue));
			break;
		default:
			break;

		}

	}

	public boolean getBooleanDefault() {
		if (type.equals(DataType.BOOL)) {
			return Boolean.valueOf(defaultValue.trim());
		}
		throw new ClassCastException(Messages.AlgorithmData_0);
	}

	@Override
	public String toString() {
		return "AlgorithmData [key=" + key + ", label=" + label + ", type=" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				+ type + ", defaultValue=" + defaultValue + ", enumOptions=" //$NON-NLS-1$ //$NON-NLS-2$
				+ Arrays.toString(enumOptions) + ", enumType=" + enumType + "]"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	/** Allowed types: String, Boolean, List<String> */
	public void setValue(Object value) {
		if ((typeTranslations.get(type)).isInstance(value)) {
			this.value = value;
		} else {
			throw new IllegalArgumentException(Messages.AlgorithmData_8 + type
					+ Messages.AlgorithmData_9 + typeTranslations.get(type)
					+ Messages.AlgorithmData_1);
		}

	}

	public Object getValue() {
		return value;
	}

}
