%%% This file is part of RefactorErl.
%%%
%%% RefactorErl is free software: you can redistribute it and/or modify
%%% it under the terms of the GNU Lesser General Public License as published
%%% by the Free Software Foundation, either version 3 of the License, or
%%% (at your option) any later version.
%%%
%%% RefactorErl is distributed in the hope that it will be useful,
%%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%% GNU Lesser General Public License for more details.
%%%
%%% You should have received a copy of the GNU Lesser General Public License
%%% along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.
%%%
%%% The Original Code is RefactorErl.
%%%
%%% The Initial Developer of the Original Code is Eötvös Loránd University.
%%% Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
%%% are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
%%% and Ericsson Hungary. All Rights Reserved.


%%% ============================================================================
%%% Module information

%%% @doc Data dependency calculation for the slicer application.
%%%
%%% @author Melinda Toth <tothmelinda@caesar.elte.hu>
-module(refsc_data_dep).

-vsn("$Rev: 17551 $ ").

-export([calc_dep_safe/1]).

-include("slicer.hrl").

calc_dep_safe(List) ->
    TabName = filename:join(mnesia:system_info(directory), "data_dep"),
    {ok, Ref} = dets:open_file(TabName, [{type, bag}]),
    dets:delete_all_objects(TabName),
    calc_dep_safe_(TabName, List),
    DepEdges = dets:match(Ref, '$1'),
    dets:close(TabName),
    lists:append(DepEdges).

calc_dep_safe_(Tab, [H | Tail]) ->
    Deps = run(fun()-> ?Dataflow:reach([H], [{back, false}]) end),
    dets:insert(Tab, [{H, E, dat_dep} || E <- Deps -- [H]]),
    [dets:insert(Tab, {E, D, dat_dep}) || E <- Deps, 
                                          D <- refcore_graph:path(E, [dep])],
    calc_dep_safe_(Tab, Tail);
calc_dep_safe_(_Tab, []) ->
    ok.

run(Fun) ->
    Self = self(),
    Pid = spawn(fun() -> R = Fun(), Self ! R end),
    receive
        Res -> Res
    after 
        10000 -> exit(Pid, kill), []
    end.
