/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation.investigations;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.forms.DetailsPart;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IDetailsPageProvider;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.MasterDetailsBlock;
import org.eclipse.ui.forms.widgets.ColumnLayout;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;

import com.refactorerl.ui.logic.investigations.Investigation;
import com.refactorerl.ui.logic.investigations.InvestigationNode;



/**
 * 
 * Implements the master side of the InvestigationMasterDetailsBlock. This side hosts a 
 * JFace tree viewer to show the investigation in it. When the selection changes
 * it notifies the details side about newly selected item.  
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 */
final class InvestigationMasterDetailsBlock extends MasterDetailsBlock {
	private FormToolkit toolkit;

	private InvestigationDetailsPage detailsPage;

	final private InvestigationEditor editor;
	

	private TreeViewer tv;
	private Investigation input; 
	
	
	public InvestigationMasterDetailsBlock(InvestigationEditor editor) {
		super();
		this.editor = editor;
	}

	@Override
	protected void registerPages(DetailsPart detailsPart) {

//		detailsPart.registerPage(InvestigationNode.class,
//				new InvestigationDetailsPage());
		detailsPart.setPageProvider(new IDetailsPageProvider() {

			@Override
			public Object getPageKey(Object object) {
				return InvestigationNode.class;
			}

			@Override
			public IDetailsPage getPage(Object key) {
				// return new InvestigationDetailsPage();
				return detailsPage;
			}
		});
	}

	@Override
	protected void createToolBarActions(IManagedForm managedForm) {
		// no toolbar
	}

	@Override
	protected void createMasterPart(final IManagedForm managedForm, Composite parent) {
		toolkit = managedForm.getToolkit();

		Section section = toolkit.createSection(parent, Section.DESCRIPTION
				| Section.TITLE_BAR);
		section.setText(Messages.InvestigationMasterDetailsBlock_0);
		section.setDescription(Messages.InvestigationMasterDetailsBlock_1);

		Composite client = toolkit.createComposite(section);
		client.setLayout(new ColumnLayout());
		section.setClient(client);

		detailsPage = new InvestigationDetailsPage(managedForm.getToolkit(), editor, this);
		
		createTreeViewer(client, new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				managedForm.fireSelectionChanged(detailsPage, event.getSelection());
			}
		});

	}

	public void setInput(Investigation inv){
		this.input = inv;
		
		tv.setInput(inv);		
		tv.expandAll();
		tv.setSelection(null); // otherwise the old node reference stays in the selection 	
	}
	
	public Investigation getInput(){
		return input;
	}
	
	public void refresh(){
		setInput(getInput());
	}

	
	private void createTreeViewer(Composite cmp, ISelectionChangedListener sl) {

		Tree tree = toolkit.createTree(cmp, SWT.VIRTUAL | SWT.BORDER);

		tv = new TreeViewer(tree);
		
		tv.setLabelProvider(new LabelProvider() {
			@Override
			public String getText(Object element) {
				InvestigationNode node = (InvestigationNode) element;
				String text = node.memo ? node.text : node.name;
				
				return text + " ("+node.label +")"; //$NON-NLS-1$ //$NON-NLS-2$
			}
		});
		
		tv.setContentProvider(new ITreeContentProvider() {

			private Investigation input;

			@Override
			public void inputChanged(Viewer viewer, Object oldInput,
					Object newInput) {			
				input = (Investigation) newInput;

			}

			@Override
			public void dispose() {
				

			}

			@Override
			public boolean hasChildren(Object element) {
				return !((InvestigationNode) element).getChildren().isEmpty();
			}

			@Override
			public Object getParent(Object element) {
				return ((InvestigationNode) element).getParent();
			}

			@Override
			public Object[] getElements(Object inputElement) {
				InvestigationNode root = input.getRoot();
				return root != null ? new Object[] {root} : new Object[0] ;
			}

			@Override
			public Object[] getChildren(Object parentElement) {
//				 System.out.println(((InvestigationNode) parentElement).getChildren()
//					.toArray());
				return ((InvestigationNode) parentElement).getChildren()
						.toArray();
			}
		});

	
//			tv.setInput(investigation);
//
//			tv.expandAll();
//			tv.getTree().setSize(500, 500);

			tv.addSelectionChangedListener(sl);


	}

}