/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.presentation.investigations;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.ManagedForm;
import org.eclipse.ui.forms.widgets.ColumnLayout;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.part.EditorPart;

import com.refactorerl.ui.communication.exceptions.CommunicationException;
import com.refactorerl.ui.communication.exceptions.ConnectionException;
import com.refactorerl.ui.communication.exceptions.RequestException;
import com.refactorerl.ui.logic.investigations.Investigation;
import com.refactorerl.ui.logic.investigations.SaveInvestigationRequest;
import com.refactorerl.ui.presentation.Activator;

/**
 * The editor part hosting an InvestigationMasterDetailsBlock. While the latter
 * is responsible for showing and manipulating an investigation, this part is
 * responsible for saving that investigation and handling the dirty (ie.
 * modified, but not saved) state.
 * 
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 */
public class InvestigationEditor extends EditorPart {

	private FormToolkit toolkit;
	private ScrolledForm form;
	private ManagedForm mf;

	private Investigation editableInput;
	private Investigation originalInput;

	private boolean dirty = false;
	private InvestigationMasterDetailsBlock md;

	@Override
	public void createPartControl(Composite parent) {
		toolkit = new FormToolkit(parent.getDisplay());
		form = toolkit.createScrolledForm(parent);
		form.setText(Messages.InvestigationEditor_0);

		form.getBody().setLayout(new ColumnLayout());
		// toolkit.createLabel(form.getBody(), "hello");

		mf = new ManagedForm(toolkit, form);

		md = new InvestigationMasterDetailsBlock(this);

		md.createContent(mf);
		md.setInput(editableInput);
		setPartName(editableInput.getName());

	}

	public void inputChanged() {		
		setDirty(!originalInput.equals(editableInput));

	}

	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}

	@Override
	protected void setInput(IEditorInput input) {
	
		super.setInput(input);

		originalInput = ((InvestigationEditorInput) input).input;
		try {
			editableInput = (Investigation) originalInput.clone();

			if (md != null)
				md.setInput(editableInput);

		} catch (CloneNotSupportedException e) {
			Activator.showErrorDialog(e, getSite().getShell());
		}
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		setSite(site);
		setInput(input);

	}

	@Override
	public void doSave(IProgressMonitor monitor) {
		doDoSave();

	}

	@Override
	public void doSaveAs() {
		InputDialog dialog = new InputDialog(getSite().getShell(), Messages.InvestigationEditor_1,
				Messages.InvestigationEditor_2, editableInput.getName(), null);
		if (dialog.open() == InputDialog.OK) {
			String name = dialog.getValue();
			editableInput.setName(name);
			doDoSave();
		}
	}

	private void doDoSave() {
		try {
			Activator.executeRequest(
					SaveInvestigationRequest.createSaveInvestigationRequest(editableInput, Activator.getProxy(), null),
					getSite().getShell());
		
			setInput(new InvestigationEditorInput(editableInput));
			setDirty(false);

		} catch (RequestException | ConnectionException | CommunicationException e) {
			Activator.showErrorDialog(e, getSite().getShell());
		}
	}

	@Override
	public boolean isSaveAsAllowed() {

		return true;
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public void setFocus() {

	}

}
