/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.logic.investigations;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import com.ericsson.otp.erlang.OtpErlangAtom;
import com.ericsson.otp.erlang.OtpErlangInt;
import com.ericsson.otp.erlang.OtpErlangList;
import com.ericsson.otp.erlang.OtpErlangObject;
import com.ericsson.otp.erlang.OtpErlangString;
import com.ericsson.otp.erlang.OtpErlangTuple;
import com.refactorerl.ui.common.OtpErlangHelper;
import com.refactorerl.ui.communication.MessageParseHelper;
import com.refactorerl.ui.communication.ReferlProxy;
import com.refactorerl.ui.communication.exceptions.AbortRequestException;
import com.refactorerl.ui.logic.AbstractRequest;

public class NodeQueryRequest extends AbstractRequest<List<OtpErlangObject>> {

	private String filePath;
	private Integer position;
	private String query;
	private OtpErlangTuple request;

	public NodeQueryRequest(String filePath, Integer position, String query,
			ReferlProxy proxy, Logger logger) {
		super(proxy, logger);
		this.filePath = filePath;
		this.position = position;
		this.query = query;
		
		OtpErlangTuple displayOpt = createDisplayOpt();
		OtpErlangTuple startOpt = createStartOpt();
		OtpErlangTuple queryStr = createQueryStr();

		request = new OtpErlangTuple(new OtpErlangObject[] {
				new OtpErlangAtom("transform"), //$NON-NLS-1$
				new OtpErlangAtom("semantic_query"), //$NON-NLS-1$
				new OtpErlangList(new OtpErlangObject[] { displayOpt, startOpt,
						queryStr }) });
		
	}

	public NodeQueryRequest(String filePath, String query, ReferlProxy proxy,
			Logger logger) {
		this(filePath, null, query, proxy, logger);
	}

	public NodeQueryRequest(String query, ReferlProxy proxy, Logger logger) {
		this(null, null, query, proxy, logger);
	}

	@Override
	protected OtpErlangObject doGetRequest() {
		return request;
	}

	@Override
	protected List<OtpErlangObject> doProcessReply(OtpErlangObject m)
			throws AbortRequestException {

		final OtpErlangObject msgData = OtpErlangHelper.nth(MessageParseHelper.getBody(m), 1);
		
		if(MessageParseHelper.isBodyAborted(m)){
			OtpErlangObject errorMessage = OtpErlangHelper.nth(OtpErlangHelper.nth(msgData, 1), 1);
			throw new AbortRequestException(OtpErlangHelper.toUnquotedString(errorMessage));
		}
		

		OtpErlangObject results = OtpErlangHelper.nth(
				msgData, 1);

	
		
		final OtpErlangObject firstResult = OtpErlangHelper.nth(OtpErlangHelper.nth(results, 0), 1);
	
		
		OtpErlangList nodesOfFirstResult = OtpErlangHelper.nth(OtpErlangHelper.nth(firstResult, 0), 1);
		
		List<OtpErlangObject> reply = new ArrayList<>();
		for (OtpErlangObject obj : nodesOfFirstResult) {
			reply.add(obj);
		}
		
		
		return reply;
	}

	private OtpErlangTuple createDisplayOpt() {
		return new OtpErlangTuple(new OtpErlangObject[] {
				new OtpErlangAtom("display_opt"), //$NON-NLS-1$
				OtpErlangHelper.createList(OtpErlangHelper.createTuple(
						new OtpErlangAtom("positions"), new OtpErlangAtom( //$NON-NLS-1$
								"scalar")), OtpErlangHelper //$NON-NLS-1$
						.createTuple(new OtpErlangAtom("output"), //$NON-NLS-1$
								new OtpErlangAtom("nodes"))) }); //$NON-NLS-1$
	}

	private OtpErlangTuple createQueryStr() {
		return OtpErlangHelper.createTuple(new OtpErlangAtom("querystr"), //$NON-NLS-1$
				new OtpErlangString(query));
	}

	private OtpErlangTuple createStartOpt() {

		OtpErlangList opts;
		if (filePath == null)
			opts = new OtpErlangList();
		else {
			OtpErlangTuple fileTuple = OtpErlangHelper.createTuple(
					new OtpErlangAtom("file"), new OtpErlangString(filePath)); //$NON-NLS-1$
			if (position == null)
				opts = OtpErlangHelper.createList(fileTuple);
			else {
				OtpErlangTuple positionTuple = OtpErlangHelper.createTuple(
						new OtpErlangAtom("position"), new OtpErlangInt( //$NON-NLS-1$
								position));
				opts = OtpErlangHelper.createList(fileTuple, positionTuple);
			}
		}
		return new OtpErlangTuple(new OtpErlangObject[] {
				new OtpErlangAtom("start_opt"), opts }); //$NON-NLS-1$
	}

	@Override
	public String toString() {
		String str = query;
		if (filePath != null)
			str += "/" + filePath.substring(filePath.lastIndexOf('/') + 1); //$NON-NLS-1$
		;
		if (position != null)
			str += "/" + position; //$NON-NLS-1$
		return str;
	}
}
