/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.logic;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.logging.Logger;

import com.ericsson.otp.erlang.OtpErlangAtom;
import com.ericsson.otp.erlang.OtpErlangInt;
import com.ericsson.otp.erlang.OtpErlangObject;
import com.ericsson.otp.erlang.OtpErlangString;
import com.ericsson.otp.erlang.OtpErlangTuple;
import com.refactorerl.ui.common.OtpErlangHelper;
import com.refactorerl.ui.common.Pair;
import com.refactorerl.ui.communication.MessageParseHelper;
import com.refactorerl.ui.communication.ReferlProxy;
import com.refactorerl.ui.communication.exceptions.ReferlException;
import com.refactorerl.ui.communication.exceptions.RequestException;
import com.refactorerl.ui.logic.queries.QueryRequest;
import com.refactorerl.ui.logic.queries.QueryResultElement;
import com.refactorerl.ui.presentation.Activator;

public class NodeTypeRequest extends AbstractRequest<NodeTypeRequest.NodeType> {

	static public enum NodeType {
		FUNCTION, OVERLOADED_FUNCTION, MACRO, RECORD, RECORD_FIELD, VARIABLE, MODULE, NONE
	}

	// see definition of reflib_ui:node_type/2
	static final private Map<String, NodeType> typeMapping;
	static {
		typeMapping = new HashMap<>();
		typeMapping.put("funappl", NodeType.FUNCTION); //$NON-NLS-1$
		typeMapping.put("recdef", NodeType.RECORD); //$NON-NLS-1$
		typeMapping.put("recfield", NodeType.RECORD_FIELD); //$NON-NLS-1$
		typeMapping.put("variable", NodeType.VARIABLE); //$NON-NLS-1$
		typeMapping.put("macrodef", NodeType.MACRO); //$NON-NLS-1$
		typeMapping.put("module", NodeType.MODULE); //$NON-NLS-1$

	}

	private String filePath;
	private Integer position;
	private OtpErlangTuple request;

	public NodeTypeRequest(String filePath, Integer position, ReferlProxy proxy, Logger logger) {
		super(proxy, logger);
		this.filePath = filePath;
		this.position = position;
		request = OtpErlangHelper.createTuple(new OtpErlangAtom("definition"), new OtpErlangString(filePath), //$NON-NLS-1$
				new OtpErlangInt(position));
	}

	@Override
	protected OtpErlangObject doGetRequest() {
		
		return request;

	}

	@Override
	protected NodeType doProcessReply(OtpErlangObject m) throws RequestException {

		if (!MessageParseHelper.isBodyOK(m))
			return NodeType.NONE;

		OtpErlangObject nodeInfo = OtpErlangHelper.nth(MessageParseHelper.getBody(m), 1);

		if (OtpErlangHelper.equalsAtom(nodeInfo, "none")) //$NON-NLS-1$
			return NodeType.NONE;

		OtpErlangObject type = OtpErlangHelper.nth(nodeInfo, 1);
		String typeStr = OtpErlangHelper.toUnquotedString(type);
		NodeType t = typeMapping.get(typeStr);

		if (t == null)
			return NodeType.NONE; 

		// check if overloaded
		if (t.equals(NodeType.FUNCTION)) {
			List<Pair<String, Integer>> funs;
			try {
				funs = new FunctionListRequest(filePath, Activator.getProxy(), null).execute();

				SortedMap<String, List<QueryResultElement>> queryResults = new QueryRequest(filePath, position, "@fun", false, //$NON-NLS-1$
						proxy, logger).execute();
				String funName = queryResults.get(queryResults.firstKey()).get(0).getValue().replaceAll("^\"|\"$", ""); //$NON-NLS-1$ //$NON-NLS-2$
				funName = funName.substring(funName.indexOf(":") + 1).split("/")[0]; //$NON-NLS-1$ //$NON-NLS-2$
				int occurences = 0;
				for (Pair<String, Integer> pair : funs) {
					if (pair.key.equals(funName))
						++occurences;
				}

				if (occurences > 1)
					t = NodeType.OVERLOADED_FUNCTION;

			} catch (ReferlException e) {
				System.out.println(e); // FIXME
			}
		}

		return t;
	}
}
