/* 
-*- coding: latin-1 -*-

This file is part of RefactorErl.

RefactorErl is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RefactorErl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.

The Original Code is RefactorErl.

The Initial Developer of the Original Code is Eötvös Loránd University.
Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
and Ericsson Hungary. All Rights Reserved.
*/

package com.refactorerl.ui.logic;

import java.util.logging.Logger;

import com.ericsson.otp.erlang.OtpErlangObject;
import com.refactorerl.ui.common.OtpErlangHelper;
import com.refactorerl.ui.communication.IReferlProgressMonitor;
import com.refactorerl.ui.communication.MessageParseHelper;
import com.refactorerl.ui.communication.ReferlProxy;
import com.refactorerl.ui.communication.exceptions.CommunicationException;
import com.refactorerl.ui.communication.exceptions.ConnectionException;
import com.refactorerl.ui.communication.exceptions.ErrorRequestException;
import com.refactorerl.ui.communication.exceptions.RequestException;


/**
 * Serves as the abstract base class for implementing new requests. It
 * implements communication with the server and logging. Extending classes must
 * implement doGetRequest() and doProcessReply(OtpErlangObject m) methods. A
 * request can be labeled modifier by overriding the isModifier() method
 * (defaults to false). RefactorErl will never run a modifier request and any
 * other kind of request concurrently. The implementor may optionally override
 * the hasProgressEvent() method to indicate whether RefactorErl will send
 * progress events while processing this request (defaults to false).
 * 
 * 
 * @author Daniel Lukacs, 2014 ELTE IK
 *
 * @param <T>
 *            Type of the return value of execute(). Extending classes are
 *            expected to define a value for this parameter.
 */
public abstract class AbstractRequest<T> {

	protected ReferlProxy proxy;

	protected Logger logger;

	public AbstractRequest(ReferlProxy proxy, Logger logger) {
		this.proxy = proxy;
		this.logger = logger;
		// this(proxy, logger, null);
	}

	/**
	 * Implementations of this method must return the OtpErlangObject request
	 * represented by the implementing request class.
	 * 
	 * @return
	 */
	protected abstract OtpErlangObject doGetRequest();

	@Override
	public String toString() {
		return doGetRequest().toString();
	}

	/**
	 * Implementations of this method must parse the full reply message and
	 * return the value which execute() should return.
	 * 
	 * @param m
	 * @return
	 * @throws RequestException
	 */
	protected abstract T doProcessReply(OtpErlangObject m) throws RequestException;

	/**
	 * Sends this request synchronously and returns with the result.
	 * 
	 * @return
	 * @throws RequestException
	 * @throws ConnectionException
	 * @throws CommunicationException
	 */
	public T execute() throws RequestException, ConnectionException, CommunicationException {
		return execute(null);
	}

	/**
	 * Sends this request synchronously and returns with the result.
	 * 
	 * @return
	 * @throws RequestException
	 * @throws ConnectionException
	 * @throws CommunicationException
	 */
	public T execute(IReferlProgressMonitor progressMonitor) throws RequestException, ConnectionException,
			CommunicationException {
		OtpErlangObject reply = proxy.send(doGetRequest(), progressMonitor);

		if (!MessageParseHelper.isBodyOK(reply)) {
			OtpErlangObject body = MessageParseHelper.getBody(reply);
			OtpErlangObject errorMessage = OtpErlangHelper.nth(OtpErlangHelper.nth(body, 1), 1);
			throw new ErrorRequestException(OtpErlangHelper.toUnquotedString(errorMessage));
		}

		return doProcessReply(reply);
	}

	/**
	 * A request can be labeled modifier by overriding this method
	 * 
	 * @return
	 */
	public boolean isModifier() {
		return false;
	}

	/**
	 * Indicates whether RefactorErl will send progress events while processing
	 * this request.
	 * 
	 * @return
	 */
	public boolean hasProgressEvent() {
		return false;
	}

}
