%%% This file is part of RefactorErl.
%%%
%%% RefactorErl is free software: you can redistribute it and/or modify
%%% it under the terms of the GNU Lesser General Public License as published
%%% by the Free Software Foundation, either version 3 of the License, or
%%% (at your option) any later version.
%%%
%%% RefactorErl is distributed in the hope that it will be useful,
%%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%% GNU Lesser General Public License for more details.
%%%
%%% You should have received a copy of the GNU Lesser General Public License
%%% along with RefactorErl.  If not, see <http://plc.inf.elte.hu/erlang/>.
%%%
%%% The Original Code is RefactorErl.
%%%
%%% The Initial Developer of the Original Code is Eötvös Loránd University.
%%% Portions created  by Eötvös Loránd University and ELTE-Soft Ltd.
%%% are Copyright 2007-2025 Eötvös Loránd University, ELTE-Soft Ltd.
%%% and Ericsson Hungary. All Rights Reserved.

%%% @doc This module implements an event listener that invalidates the database
%%% when a file was added, dropped or reloaded.
%%%
%%% @author Csaba Hoch <hoch@inf.elte.hu>

-module(cl_db_invalidator).
-vsn("$Rev: 17551 $").
-behaviour(gen_event).

-export([start/0, stop/0]).

-export([init/1, handle_event/2, handle_call/2, handle_info/2,
         terminate/2, code_change/3]).

%% @doc Adds the event handler to the event manager of reflib_ui.
start() ->
    reflib_ui:add_msg_handler(?MODULE, null).

%% @doc Removes the event handler from the event manager of reflib_ui.
stop() ->
    reflib_ui:del_msg_handler(?MODULE, null).

%% ==================================================================
%% Event handler callbacks

%% @private
init(State) ->
    {ok, State}.

%% @private
handle_event(Event, State) ->
    case Event of
        {E, _} when E == drop; E == add; E == reload ->
            cl_db:invalidate();
        _ ->
            ok
    end,
    {ok, State}.

%% @private
handle_call(_Req, State) ->
    {ok, undefined, State}.

%% @private
handle_info(_Info, State) ->
    {ok, State}.

%% @private
terminate(_Arg, _State) ->
    ok.

%% @private
code_change(_, S, _) ->
    {ok, S}.
